open Printf

let test_location = "test/"

type test_status = Pass | Fail
type color = Red | Green

let colorize msg c =
  let pad = match c with
      Red -> "31"
    | Green -> "32" in
  let template = format_of_string "
    \027[%sm%s" in
  printf template pad msg;
  flush stdout;
;;

let run_cmd cmd =
  let chan = Unix.open_process_in cmd in
  let res = ref ([] : string list) in
  let rec helper () =
    let line = input_line chan in
    res := line :: !res;
    helper () in
  try helper ()
  with End_of_file ->
    let status = Unix.close_process_in chan in
    let cmd_result = match status with
        Unix.WEXITED(c) -> if c == 0 then Pass else Fail
      | _ -> Fail in
    (List.rev !res, cmd_result)
;;

let diff_output lines filename =
  let dump_to_file lines fname =
    let oc = open_out fname in
    List.iter (fun line -> fprintf oc "%s\n" line) lines;
    close_out oc in
  let _ = dump_to_file lines "test_output.txt" in
  let cmd = sprintf "diff -B test_output.txt %s" filename in
  let diff_output, status = run_cmd cmd in
  begin
    match status with
      Pass -> None
    | Fail -> Some(String.concat "\n" diff_output)
  end
;;

let run_testcase fname =
  let test_type, test_name =
    match (Str.split (Str.regexp "-") fname) with
      "fail" :: x :: [] -> Fail, x
    | "pass" :: x :: [] -> Pass, x
    | _ -> raise (Failure ("Invalid file format - " ^ fname ^ ". Must have only one '-'")) in

  let fpath = Filename.concat test_location fname in
  let cmd = sprintf "./spy.out %s" fpath in

  let output_filename = Str.replace_first (Str.regexp "spy") "txt" fname in
  let output_path = Filename.concat test_location output_filename in

  let cmd_output, status = run_cmd cmd in
  match test_type, status with
      Pass, Pass -> begin
        let node_output, status = run_cmd "node out.js" in
        (match diff_output node_output output_path with
            None -> colorize (sprintf "✓ " ^ fname) Green; Pass
          | Some(op) -> begin
              colorize (sprintf "✖ %s" fname) Red;
              colorize (sprintf "%s\n\n" op) Red;
            end; Fail)
      end

    | Fail, Fail ->
      (match diff_output cmd_output output_path with
          None -> colorize (sprintf "✓ %s" fname) Green; Pass
        | Some(op) -> begin
            colorize (sprintf "✖ %s" fname) Red;
            colorize (sprintf "%s\n\n" op) Red;
          end; Fail)

    | Pass, Fail -> begin
        colorize (sprintf "✖ %s" fname) Red;
        colorize "Expected test case to pass, but it failed" Red;
      end; Fail

    | Fail, Pass -> begin
        colorize (sprintf "✖ %s" fname) Red;
        colorize "Expected test case to fail, but it passed" Red;
      end; Fail
;;

let run testcases () =
  let total = List.length testcases in
  let passing = List.fold_left (fun acc t -> acc + (match run_testcase t with Pass -> 1 | Fail -> 0)) 0 testcases in
  let template = format_of_string "\027[37m

    Test Summary
    -------------------------
    All testcases complete.
    Total Testcases : %d
    Total Passing   : %d
    Total Failed    : %d
      \n" in
  let failures = total - passing in
  Printf.printf template total passing failures;
  if failures = 0 then Pass else Fail
;;

(* returns a list of file names in a directory *)
let get_files dirname =
    let d = Unix.opendir dirname in
    let files = ref ([] : string list) in
    let rec helper () =
      let fname = Unix.readdir d in
      files := fname :: !files;
      helper () in
    try helper () with End_of_file -> Unix.closedir d; files
;;

let init () =
  let files = !(get_files test_location) in
  let testcases = List.filter
      (fun f ->
         try ignore (Str.search_forward (Str.regexp ".spy") f 0); true
         with Not_found -> false)
      files in
  match (run testcases ()) with 
    Pass -> exit 0
  | Fail -> exit 1
;;

init ();

